#!/bin/bash

YELLOW='\033[0;33m'
GREEN='\033[0;32m'
RED='\033[0;31m'
NC='\033[0m'

spinner() {
    local pid=$1
    local delay=0.1
    local spinstr='|/-\'
    while kill -0 $pid 2>/dev/null; do
        local temp=${spinstr#?}
        printf " ${YELLOW}[%c]${NC}  " "$spinstr"
        spinstr=$temp${spinstr%"$temp"}
        sleep $delay
        printf "\b\b\b\b\b\b"
    done
    printf "     \b\b\b\b\b"
}

run_with_spinner() {
    local full_command="$*"

    echo "Executing: ${full_command}..."

    "$@" &
    local cmd_pid=$!

    spinner "$cmd_pid"
    wait "$cmd_pid"

    if [ $? -eq 0 ]; then
        echo -e "${GREEN}Command successed: ${full_command}${NC}"
    else
        echo -e "${RED}Command error: ${full_command}${NC}"
        exit 1
    fi
}

restore_dump() {
    local dump_file="$1"

    local base_name=$(basename "$dump_file")
    local table_name="${base_name%.sql}"

    echo "Restoring table '$table_name' from dump '$dump_file'..."

    PGPASSWORD="$password" pg_restore -d "$psql_db_name" -U "$user" -h "$host" -p "$psql_port" "$dump_file" &
    local restore_pid=$!

    spinner "$restore_pid"
    wait "$restore_pid"

    if [ $? -eq 0 ]; then
        echo -e "${GREEN}Table '$table_name' was restored successfully from '$dump_file'${NC}"
    else
        echo -e "${RED}Error restoring table '$table_name' from '$dump_file'${NC}"
        exit 1
    fi
}

restore_sql_file() {
    local sql_file="$1"

    local base_name=$(basename "$sql_file")
    local table_name="${base_name%.sql}"

    echo "Restoring table '$table_name' from sql file '$sql_file'..."

    PGPASSWORD="$password" psql -U "$user" -d "$psql_db_name" -h "$host" -p "$psql_port" -1 -f "$sql_file" --single-transaction &
    local psql_pid=$!

    spinner "$psql_pid"
    wait "$psql_pid"

    if [ $? -eq 0 ]; then
        echo -e "${GREEN}Table '$table_name'  was restored successfully from '$sql_file'${NC}"
    else
        echo -e "${RED}Error restoring table '$table_name' from '$sql_file'${NC}"
        exit 1
    fi
}

host=$1
user=$2
password=$3
psql_db_name=$4
psql_port=$5
ch_def_user_password=$6
ch_db_name=$7

if [ "$host" == "0.0.0.0" ] || [ "$host" == "localhost" ];
then
    python3 src/py_utils/service.py evi-scud stop
    if [ $? -eq 0 ]
    then
        echo "evi-scud stopped"
    else
        exit 1
    fi

    python3 src/py_utils/service.py evi-core stop
    if [ $? -eq 0 ]
    then
        echo "evi-core stopped"
    else
        exit 1
    fi

    python3 src/py_utils/service.py evi-live stop
    if [ $? -eq 0 ]
    then
        echo "evi-live stopped"
    else
        exit 1
    fi

    python3 src/py_utils/service.py evi-analyzer stop
    if [ $? -eq 0 ]
    then
        echo "evi-analyzer stopped"
    else
        exit 1
    fi

    python3 src/py_utils/service.py evi-archive stop
    if [ $? -eq 0 ]
    then
        echo "evi-archive stopped"
    else
        exit 1
    fi

    python3 src/py_utils/service.py postgresql restart
    if [ $? -eq 0 ]
    then
        echo "postgresql restarted"
    else
        exit 1
    fi

    python3 src/py_utils/service.py clickhouse-server restart
    if [ $? -eq 0 ]
    then
        echo "clickhouse restarted"
    else
        exit 1
    fi

    su postgres -c "psql -c \"create user $user;\""
    su postgres -c "psql -c \"ALTER USER $user WITH PASSWORD '$password';\""
    su postgres -c "psql -c \"ALTER USER $user WITH SUPERUSER;\""

    su postgres -c "psql -c \"DROP DATABASE IF EXISTS $psql_db_name;\""
    su postgres -c "psql -c \"CREATE DATABASE $psql_db_name;\""

    echo "PostgreSQL configured"

    echo clickhouse-client --user=default --password=$ch_def_user_password --query=\"CREATE USER IF NOT EXISTS $user IDENTIFIED WITH sha256_password BY \'$password\'\" --query=\"GRANT CURRENT GRANTS on \*.\* TO $user WITH GRANT OPTION\"
    clickhouse-client --user=default --password=$ch_def_user_password --query="CREATE USER IF NOT EXISTS $user IDENTIFIED WITH sha256_password BY '$password'" --query="GRANT CURRENT GRANTS on *.* TO $user WITH GRANT OPTION"
    echo clickhouse-client --user=$user --password=$password --query=\"DROP DATABASE IF EXISTS $ch_db_name\"
    clickhouse-client --user=$user --password=$password --query="DROP DATABASE IF EXISTS $ch_db_name"
    echo clickhouse-client --user=$user --password=$password --query=\"CREATE DATABASE IF NOT EXISTS $ch_db_name\"
    clickhouse-client --user=$user --password=$password --query="CREATE DATABASE IF NOT EXISTS $ch_db_name"

    echo "Clickhouse configured"
fi

PGPASSWORD=$password psql -X -h $host -U $user -d $psql_db_name -p $psql_port -c "CREATE SCHEMA IF NOT EXISTS evi;"

clickhouse-client --user=$user --password=$password --query="CREATE TABLE $ch_db_name.operational_events
(
    id UUID,
    timestamp DateTime64(6),
    code Int16,
    level Int16,
    controller String,
    location String,
    user String,
    relay String,
    di String,
    camera String,
    cameraStream String,
    card String,
    car String,
    worker String,
    guest String,
    prevCard String,
    prevWorker String,
    prevGuest String,
    prevDi String,
    service String,
    updateTask String
)
ENGINE = ReplacingMergeTree
PARTITION BY toYYYYMM(timestamp)
ORDER BY (timestamp, code, level, controller, location, user, relay, di, camera, cameraStream, card, car, worker, guest, prevCard, prevWorker, prevGuest, prevDi, service, updateTask)
SETTINGS index_granularity = 8192;"

rm -rf restores
mkdir restores -p

run_with_spinner python3 src/py_utils/preconf.py ./restores/preconf.sql
restore_sql_file "./restores/preconf.sql"

run_with_spinner python3 src/py_utils/gen_users_dump.py ./dumps/users.csv ./restores/users.sql
restore_sql_file "./restores/users.sql"

restore_dump "./dumps/apikey.sql"
restore_dump "./dumps/token.sql"
restore_dump "./dumps/organizations.sql"
restore_dump "./dumps/divisions.sql"
restore_dump "./dumps/positions.sql"
restore_dump "./dumps/location.sql"

run_with_spinner python3 src/py_utils/gen_camera_dump.py ./dumps/camera.csv ./restores/camera.sql
restore_sql_file "./restores/camera.sql"

restore_dump "./dumps/camera_stream.sql"
restore_dump "./dumps/camera_archive_schedule.sql"
restore_dump "./dumps/views.sql"
restore_dump "./dumps/access_views.sql"
restore_dump "./dumps/cell_views.sql"
restore_dump "./dumps/photo.sql"
restore_dump "./dumps/car.sql"
restore_dump "./dumps/access_level_worker.sql"

run_with_spinner python3 src/py_utils/gen_workers_dump.py ./dumps/worker.csv ./restores/worker.sql
restore_sql_file "./restores/worker.sql"

restore_dump "./dumps/worker_car.sql"
restore_dump "./dumps/worker_urv.sql"
restore_dump "./dumps/pass_info.sql"
restore_dump "./dumps/calendar.sql"
restore_dump "./dumps/calendar_day_time.sql"
restore_dump "./dumps/calendar_dates.sql"
restore_dump "./dumps/access_level_controller.sql"
restore_dump "./dumps/controller.sql"
restore_dump "./dumps/relay.sql"
restore_dump "./dumps/reader.sql"

run_with_spinner python3 src/py_utils/gen_discrete_input_dump.py ./dumps/discrete_input.csv ./restores/discrete_input.sql
restore_sql_file "./restores/discrete_input.sql"

restore_dump "./dumps/access_pattern.sql"
restore_dump "./dumps/access_pattern_relay.sql"
restore_dump "./dumps/card.sql"
restore_dump "./dumps/worker_card.sql"
restore_dump "./dumps/worker_calendar.sql"
restore_dump "./dumps/worker_access_pattern.sql"

run_with_spinner python3 src/py_utils/gen_operational_events_dump.py ./dumps/operational_event.csv ./restores/critical_event.sql ./restores/operational_event.csv
restore_sql_file "./restores/critical_event.sql"
run_with_spinner bash -c "clickhouse-client --user=$user --password=$password --query=\"INSERT INTO $ch_db_name.operational_events FORMAT CSV\" < ./restores/operational_event.csv"

run_with_spinner python3 src/py_utils/gen_pass_event_dump.py ./dumps/pass_event.csv ./restores/pass_event
for file in ./restores/pass_event_*.sql; do
    restore_sql_file "$file"
done

run_with_spinner python3 src/py_utils/gen_guest_pass_event_dump.py ./dumps/guest_pass_event.csv ./restores/guest_pass_event.sql
restore_sql_file "./restores/guest_pass_event.sql"

restore_dump "./dumps/user_log.sql"
restore_dump "./dumps/settings.sql"

run_with_spinner python3 src/py_utils/gen_guest_dump.py ./dumps/guest.csv ./restores/guest.sql
restore_sql_file "./restores/guest.sql"

run_with_spinner python3 src/py_utils/gen_visit_dump.py ./dumps/visit.csv ./restores/visit.sql
restore_sql_file "./restores/visit.sql"

restore_dump "./dumps/visit_card.sql"
restore_dump "./dumps/visit_card_history.sql"
restore_dump "./dumps/interactive_plan.sql"
restore_dump "./dumps/entity_plan_position.sql"
restore_dump "./dumps/license_params.sql"
restore_dump "./dumps/faces.sql"
PGPASSWORD="$password" psql -U "$user" -d "$psql_db_name" -h "$host" -p "$psql_port" -c "ALTER TABLE faces SET SCHEMA "evi";"

run_with_spinner python3 src/py_utils/gen_streams_dump.py ./dumps/streams.csv ./restores/streams.sql
restore_sql_file "./restores/streams.sql"

run_with_spinner python3 src/py_utils/gen_stream_params_dump.py ./dumps/stream_params.csv ./restores/stream_params.sql
restore_sql_file "./restores/stream_params.sql"

run_with_spinner python3 src/py_utils/gen_analytics_schedule_dump.py ./dumps/streams.csv ./restores/analytics_schedule.sql
restore_sql_file "./restores/analytics_schedule.sql"

run_with_spinner python3 src/py_utils/gen_points_zones_dump.py ./dumps/points.csv ./dumps/zones.csv ./restores/points_zones.sql
restore_sql_file "./restores/points_zones.sql"

run_with_spinner python3 src/py_utils/gen_triggers_dump.py ./dumps/camera.csv ./dumps/actions.csv ./dumps/zones.csv ./restores/triggers.sql
restore_sql_file "./restores/triggers.sql"

if [ "$host" == "127.0.0.1" ]
then
    python3 src/py_utils/service.py evi-scud start
    if [ $? -eq 0 ]
    then
        echo "evi-scud started"
    else
        exit 1
    fi

    python3 src/py_utils/service.py evi-core start
    if [ $? -eq 0 ]
    then
        echo "evi-core started"
    else
        exit 1
    fi

    python3 src/py_utils/service.py evi-live start
    if [ $? -eq 0 ]
    then
        echo "evi-live started"
    else
        exit 1
    fi

    python3 src/py_utils/service.py evi-analyzer start
    if [ $? -eq 0 ]
    then
        echo "evi-analyzer started"
    else
        exit 1
    fi

    python3 src/py_utils/service.py evi-archive start
    if [ $? -eq 0 ]
    then
        echo "evi-archive started"
    else
        exit 1
    fi
fi

exit 0
